/*
Objet :
	Création des tables vides permettant l'import des données DVF selon le modèle défini par l'ADEF (hors tables annexes)
Auteur : 
	CEREMA "PCI Foncier et Stratégies Foncières"
	Direction Territoriale Nord-Picardie pour le compte EPF Nord Pas de Calais / DGALN	
	Antoine Herman
	Jérôme Douché
	
Droits :	
	Ce script est fourni sous licence Creative Commons 3.0 CC-BY-SA
	http://creativecommons.org/licenses/by-sa/3.0/fr/
	
Version 1.2 - Mars 2015 :
		- suppression d'une virgule
		- précision de l'encodage
		- variable 'code_departement' de la table 'parcelle' : varchar(2) >> varchar(3)		
		- variable 'no_volume' de la table 'volume' : varchar(5) >> varchar(7)		
		
*/

SET client_encoding = 'UTF8';

-----------------------------------
-- Création du schéma "DVF"
-----------------------------------
DROP SCHEMA IF EXISTS dvf CASCADE;
CREATE SCHEMA dvf;

-----------------------------------
-- Tables annexes
-----------------------------------
-- table ann_nature_culture
DROP TABLE IF EXISTS dvf.ann_nature_culture;
CREATE TABLE dvf.ann_nature_culture
(
	nature_culture character varying(2),
	lib_nature_culture character varying(254),
	CONSTRAINT ann_nature_culture_pkey PRIMARY KEY (nature_culture)	
);
COMMENT ON TABLE dvf.ann_nature_culture IS 'Table annexe contenant les natures de culture';



-- table ann_nature_culture_speciale
DROP TABLE IF EXISTS dvf.ann_nature_culture_speciale;
CREATE TABLE dvf.ann_nature_culture_speciale
(
	nature_culture_speciale character varying(5),
	lib_nature_culture_speciale character varying(254),
	CONSTRAINT nature_culture_speciale_pkey PRIMARY KEY (nature_culture_speciale)
);
COMMENT ON TABLE dvf.ann_nature_culture_speciale IS 'Table annexe contenant les natures de culture spéciale';

-- table ann_cgi
DROP TABLE IF EXISTS dvf.ann_cgi;
CREATE TABLE dvf.ann_cgi
(
	id_article_cgi serial,
	article_cgi character varying(20),
	lib_article_cgi character varying(254),
	CONSTRAINT ann_cgi_pkey PRIMARY KEY (id_article_cgi),
	CONSTRAINT ann_cgi_unique UNIQUE (article_cgi)
);
COMMENT ON TABLE dvf.ann_cgi IS 'Table annexe contenant les articles cgi';



-- table ann_nature_mutation
DROP TABLE IF EXISTS dvf.ann_nature_mutation;
CREATE TABLE dvf.ann_nature_mutation
(
	id_nature_mutation serial,
	lib_nature_mutation character varying(254),
	CONSTRAINT ann_nature_mutation_pkey PRIMARY KEY (id_nature_mutation),
	CONSTRAINT nature_mutation_unique UNIQUE (lib_nature_mutation)
);
COMMENT ON TABLE dvf.ann_nature_mutation IS 'Table annexe contenant les natures de mutation';



-- table ann_type_local
DROP TABLE IF EXISTS dvf.ann_type_local;
	--> création de la table
CREATE TABLE dvf.ann_type_local
(
	code_type_local integer,
	lib_type_local character varying(254),
	CONSTRAINT ann_type_local_pkey PRIMARY KEY (code_type_local)
);
COMMENT ON TABLE dvf.ann_type_local IS 'Table annexe contenant les types de locaux';


	--> ajout des données
INSERT INTO dvf.ann_type_local(code_type_local, lib_type_local) VALUES (1, 'Maison');
INSERT INTO dvf.ann_type_local(code_type_local, lib_type_local) VALUES (2, 'Appartement');
INSERT INTO dvf.ann_type_local(code_type_local, lib_type_local) VALUES (3, 'Dépendance');
INSERT INTO dvf.ann_type_local(code_type_local, lib_type_local) VALUES (4, 'Local industriel, commercial ou assimilé');


-----------------------------------
-- Tables de données
-----------------------------------

-- table mutation
DROP TABLE IF EXISTS dvf.mutation;
CREATE TABLE dvf.mutation
(
	id_mutation serial,
	code_service_ch varchar(7),
	reference_document varchar(10),
	date_mutation date,
	id_nature_mutation integer,
	CONSTRAINT mutation_pkey PRIMARY KEY (id_mutation),
	CONSTRAINT mutation_unique UNIQUE (code_service_ch, reference_document)
);
COMMENT ON TABLE dvf.mutation IS 'Table des mutations';


-- table mutation_article_cgi
DROP TABLE IF EXISTS dvf.mutation_article_cgi;
CREATE TABLE dvf.mutation_article_cgi
(
	id_mutation integer,
	id_article_cgi integer,
	ordre_article_cgi integer,
	CONSTRAINT mutation_article_cgi_pkey PRIMARY KEY (id_mutation,id_article_cgi)
	--CONSTRAINT mutation_article_cgi_unique UNIQUE (id_mutation, id_article_cgi,ordre_article_cgi)
);
COMMENT ON TABLE dvf.mutation_article_cgi IS 'Table des articles du code général des impôts (cgi) attachés à la mutation';


-- table disposition
DROP TABLE IF EXISTS dvf.disposition;
CREATE TABLE dvf.disposition
(
	id_disposition serial,
	id_mutation integer,
	no_disposition integer,
	valeur_fonciere numeric,
	nombre_de_lots integer[],
	CONSTRAINT disposition_pkey PRIMARY KEY (id_disposition),
	CONSTRAINT disposition_unique UNIQUE (id_mutation, no_disposition)
);
COMMENT ON TABLE dvf.disposition IS 'Table des dispositions';

	
-- table disposition_parcelle
DROP TABLE IF EXISTS dvf.disposition_parcelle;
CREATE TABLE dvf.disposition_parcelle
(
	id_dispoparc serial,
	id_disposition integer,
	id_parcelle integer,
	CONSTRAINT disposition_parcelle_pkey PRIMARY KEY (id_dispoparc),
	CONSTRAINT disposition_parcelle_unique UNIQUE (id_disposition,id_parcelle)
);
COMMENT ON TABLE dvf.disposition_parcelle IS 'Table des parcelles attachées à la disposition';


-- table parcelle
DROP TABLE IF EXISTS dvf.parcelle;
CREATE TABLE dvf.parcelle
(
	id_parcelle serial,
	idpar varchar(14),
	code_departement varchar(3),
	code_commune varchar(3),
	prefixe_de_section varchar(3),
	section varchar(2),
	no_plan varchar(4),
	CONSTRAINT parcelle_pkey PRIMARY KEY (id_parcelle),
	CONSTRAINT parcelle_unique UNIQUE (idpar)
);
CREATE INDEX idpar_btree_idx ON dvf.parcelle USING BTREE(idpar);
COMMENT ON TABLE dvf.parcelle IS 'Table des parcelles';


-- table lot
DROP TABLE IF EXISTS dvf.lot;
CREATE TABLE dvf.lot
(
	id_lot serial,
	id_dispoparc integer,
	id_local integer,
	id_suf integer,
	id_volume integer,
	no_lot integer,
	surface_carrez numeric,
	CONSTRAINT lot_pkey PRIMARY KEY (id_lot),
	CONSTRAINT lot_unique UNIQUE (id_dispoparc, id_local, id_suf, id_volume, no_lot, surface_carrez)
);
COMMENT ON TABLE dvf.lot IS 'Table des lots (seuls les 5 premiers lots sont mentionnés)';


-- table local
DROP TABLE IF EXISTS dvf.local;
CREATE TABLE dvf.local
(
	id_local serial,
	id_dispoparc integer,
	identifiant_local varchar(7),
	code_type_local integer,
	nombre_pieces_principales integer,
	surface_reelle_bati numeric,
	CONSTRAINT local_pkey PRIMARY KEY (id_local),
	CONSTRAINT local_unique UNIQUE (id_dispoparc,identifiant_local)
);
COMMENT ON TABLE dvf.local IS 'Table des locaux';


-- table suf
DROP TABLE IF EXISTS dvf.suf;
CREATE TABLE dvf.suf
(
	id_suf serial,
	id_dispoparc integer,
	nb_suf_idt integer,
	surface_terrain numeric,
	nature_culture varchar(2),
	nature_culture_speciale varchar(5),
	idsuf_tmp varchar(18),
	CONSTRAINT suf_pkey PRIMARY KEY (id_suf),
	CONSTRAINT suf_unique UNIQUE (id_dispoparc,nb_suf_idt, idsuf_tmp)
);
COMMENT ON TABLE dvf.suf IS 'Table des subdivisions fiscales (suf)';


-- table volume
DROP TABLE IF EXISTS dvf.volume;
CREATE TABLE dvf.volume
(
	id_volume serial,
	id_dispoparc integer,
	no_volume varchar(7),
	CONSTRAINT volume_pkey PRIMARY KEY (id_volume),
	CONSTRAINT volume_unique UNIQUE (id_dispoparc, no_volume)
);
COMMENT ON TABLE dvf.volume IS 'Table des volumes (division de l’espace situé au-dessus et en dessous d’un terrain pour certaines co-propriétés verticales)';


-- table adresse
DROP TABLE IF EXISTS dvf.adresse;
CREATE TABLE dvf.adresse
(
	id_adresse serial,
	no_voie integer,
	b_t_q varchar(1),
	code_voie varchar(4),
	type_de_voie varchar(4),
	voie varchar(254),
	code_postal varchar(5),
	commune varchar(254),
	idadr_tmp varchar(532),
	CONSTRAINT adresse_pkey PRIMARY KEY (id_adresse),
	CONSTRAINT adresse_unique UNIQUE (idadr_tmp)
);
CREATE INDEX id_adresse_idadr_tmp_btree_idx ON dvf.adresse USING BTREE(idadr_tmp);
COMMENT ON TABLE dvf.adresse IS 'Table contenant les adresses (provenant des parcelles et des locaux)';




-- table adresse_dispoparc
DROP TABLE IF EXISTS dvf.adresse_dispoparc;
CREATE TABLE dvf.adresse_dispoparc
(
	id_adresse integer,
	id_dispoparc integer,
	CONSTRAINT adresse_dispoparc_pkey PRIMARY KEY (id_adresse,id_dispoparc)
);
COMMENT ON TABLE dvf.adresse_dispoparc IS 'Table de liaison entre la table ‘adresse’ et la table ‘disposition_parcelle’';


-- table adresse_local
DROP TABLE IF EXISTS dvf.adresse_local;
CREATE TABLE dvf.adresse_local
(
	id_adresse integer,
	id_local integer,
	CONSTRAINT adresse_local_pkey PRIMARY KEY (id_adresse,id_local)
);
COMMENT ON TABLE dvf.adresse_local IS 'Table de liaison entre la table ‘adresse’ et la table ‘local’';

-----------------------------------
-- Ajout des commentaires sur les champs
-----------------------------------
COMMENT ON COLUMN dvf.adresse.id_adresse IS 'Identifiant de la table adresse';
COMMENT ON COLUMN dvf.adresse.no_voie IS 'Numéro dans la voie';
COMMENT ON COLUMN dvf.adresse.b_t_q IS 'Indice de répétition';
COMMENT ON COLUMN dvf.adresse.code_voie IS 'Code de la voie';
COMMENT ON COLUMN dvf.adresse.type_de_voie IS 'Exemple : Rue, Avenue, etc.';
COMMENT ON COLUMN dvf.adresse.voie IS 'Libellé de la voie';
COMMENT ON COLUMN dvf.adresse.code_postal IS 'Code postal';
COMMENT ON COLUMN dvf.adresse.commune IS 'Libellé de la commune';
COMMENT ON COLUMN dvf.adresse.idadr_tmp IS 'Identifiant de la table adresse temporaire facilitant l’import des nouvelles données';
COMMENT ON COLUMN dvf.adresse_dispoparc.id_adresse IS 'Identifiant de la table adresse';
COMMENT ON COLUMN dvf.adresse_dispoparc.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.adresse_local.id_adresse IS 'Identifiant de la table adresse';
COMMENT ON COLUMN dvf.adresse_local.id_local IS 'Identifiant de la table local';
COMMENT ON COLUMN dvf.ann_nature_culture.nature_culture IS 'Groupe de nature de culture';
COMMENT ON COLUMN dvf.ann_nature_culture.lib_nature_culture IS 'Libellé de nature de culture';
COMMENT ON COLUMN dvf.ann_nature_culture_speciale.nature_culture_speciale IS 'Groupe de nature de culture spéciale';
COMMENT ON COLUMN dvf.ann_nature_culture_speciale.lib_nature_culture_speciale IS 'Libellé de nature de culture spéciale';
COMMENT ON COLUMN dvf.ann_nature_mutation.id_nature_mutation IS 'Identifiant de la table ann_nature_mutation';
COMMENT ON COLUMN dvf.ann_nature_mutation.lib_nature_mutation IS 'Libellé de la nature de mutation';
COMMENT ON COLUMN dvf.ann_type_local.code_type_local IS 'Code du type de local';
COMMENT ON COLUMN dvf.ann_type_local.lib_type_local IS 'Libellé du type de local';
COMMENT ON COLUMN dvf.ann_cgi.id_article_cgi IS 'Identifiant de la table article_cgi';
COMMENT ON COLUMN dvf.ann_cgi.article_cgi IS 'Code de l’article_cgi';
COMMENT ON COLUMN dvf.ann_cgi.lib_article_cgi IS 'Libellé de l’article_cgi';
COMMENT ON COLUMN dvf.disposition.id_disposition IS 'Identifiant de la table disposition';
COMMENT ON COLUMN dvf.disposition.id_mutation IS 'Identifiant de la table mutation';
COMMENT ON COLUMN dvf.disposition.no_disposition IS 'Numéro de disposition';
COMMENT ON COLUMN dvf.disposition.valeur_fonciere IS 'Prix ou évaluation déclaré dans le cadre d’une mutation à titre onéreux';
COMMENT ON COLUMN dvf.disposition.nombre_de_lots IS 'Nombre de lots';
COMMENT ON COLUMN dvf.disposition_parcelle.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.disposition_parcelle.id_disposition IS 'Identifiant de la table disposition';
COMMENT ON COLUMN dvf.disposition_parcelle.id_parcelle IS 'Identifiant de la table parcelle';
COMMENT ON COLUMN dvf.local.id_local IS 'Identifiant de la table local';
COMMENT ON COLUMN dvf.local.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.local.identifiant_local IS 'Identifiant du local';
COMMENT ON COLUMN dvf.local.code_type_local IS 'Code du type de local';
COMMENT ON COLUMN dvf.local.nombre_pieces_principales IS 'Nombre de pièces principales';
COMMENT ON COLUMN dvf.local.surface_reelle_bati IS 'Surface réelle attachée à l’identifiant local (somme de la surface réelle du local et des surfaces des dépendances)';
COMMENT ON COLUMN dvf.lot.id_lot IS 'Identifiant de la table lot';
COMMENT ON COLUMN dvf.lot.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.lot.id_local IS 'Identifiant de la table local';
COMMENT ON COLUMN dvf.lot.id_suf IS 'Identifiant de la table suf';
COMMENT ON COLUMN dvf.lot.id_volume IS 'Identifiant de la table volume';
COMMENT ON COLUMN dvf.lot.no_lot IS 'Numéro du lot';
COMMENT ON COLUMN dvf.lot.surface_carrez IS 'Surface loi Carrez du lot';
COMMENT ON COLUMN dvf.mutation.id_mutation IS 'Identifiant de la table mutation';
COMMENT ON COLUMN dvf.mutation.code_service_ch IS 'Code du service de conservation des hypothèques ';
COMMENT ON COLUMN dvf.mutation.reference_document IS 'Références d’enregistrement du document';
COMMENT ON COLUMN dvf.mutation.date_mutation IS 'Date de signature du document';
COMMENT ON COLUMN dvf.mutation.id_nature_mutation IS 'Identifiant de la table ann_nature_mutation';
COMMENT ON COLUMN dvf.mutation_article_cgi.id_mutation IS 'Identifiant de la table mutation';
COMMENT ON COLUMN dvf.mutation_article_cgi.id_article_cgi IS 'Identifiant de l’article cgi';
COMMENT ON COLUMN dvf.mutation_article_cgi.ordre_article_cgi IS 'Numéro de l’ordre de l’article cgi';
COMMENT ON COLUMN dvf.parcelle.id_parcelle IS 'Identifiant de la table parcelle';
COMMENT ON COLUMN dvf.parcelle.idpar IS 'Identifiant de la parcelle';
COMMENT ON COLUMN dvf.parcelle.code_departement IS 'Code du département';
COMMENT ON COLUMN dvf.parcelle.code_commune IS 'Code Insee de la commune';
COMMENT ON COLUMN dvf.parcelle.prefixe_de_section IS 'Préfixe de section de la parcelle';
COMMENT ON COLUMN dvf.parcelle.section IS 'Numéro de la section de la parcelle';
COMMENT ON COLUMN dvf.parcelle.no_plan IS 'Numéro de la parcelle';
COMMENT ON COLUMN dvf.suf.id_suf IS 'Identifiant de la table suf';
COMMENT ON COLUMN dvf.suf.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.suf.nb_suf_idt IS 'Nombre de suf identiques';
COMMENT ON COLUMN dvf.suf.surface_terrain IS 'Contenance du terrain (en m2)';
COMMENT ON COLUMN dvf.suf.nature_culture IS 'Libellé de nature de culture';
COMMENT ON COLUMN dvf.suf.nature_culture_speciale IS 'Groupe de nature de culture spéciale';
COMMENT ON COLUMN dvf.suf.idsuf_tmp IS 'Identifiant de la table suf temporaire facilitant l’import des nouvelles données';
COMMENT ON COLUMN dvf.volume.id_volume IS 'Identifiant de la table volume';
COMMENT ON COLUMN dvf.volume.id_dispoparc IS 'Identifiant de la table disposition_parcelle';
COMMENT ON COLUMN dvf.volume.no_volume IS 'Numéro de volume';



